{******************************************************}
{            Slim Dynamic Array Class v3.0             }
{        Copyright 1997 RealSoft Development           }
{           support:  www.realsoftdev.com              }
{                    ------------                      }
{ This is a slim version on the Dynarray Class         }
{ to be distributed with your source code.  RealSoft   }
{ grants an unrestricted license to include this unit  }
{ in its un modified format.  For a full featured      }
{ version, contact dan@realSoftdev.com, or visit the   }
{ Compuserve Delphi forum.  Do not remove this notice. }
{******************************************************}

unit rsDynslm;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Dialogs;

type
  EDynArrayException    = class(Exception);
  EDynArrayCreateError  = class(EDynArrayException);
  EDynArrayIndexBounds  = class(EDynArrayException);
  EDynArrayResources    = class(EDynArrayException);
  EDynArrayFileError    = class(EDynArrayException);
  EDynArrayAddError     = class(EDynArrayException);
  EDynArrayInsError     = class(EDynArrayException);
  EDynArrayDelError     = class(EDynArrayException);

  TrsDynArray = class
    private
      FPointer     : Pointer;
      FItemSize    : Longint;
      FItemCount   : Longint;
      FArraySize   : LongInt;
      function  GetItems (Index : Longint) : pointer;
    public
      constructor Create ( ItemSize : Longint );
      destructor  Destroy; override;
      function    Clear  : Pointer;
      function    Add    ( const Item ) : Pointer;
      function    Insert ( Index: Longint; const Item ) : Pointer;
      function    Delete ( Index : Longint  ) : Pointer;
      function    SaveToFile   ( filename : String ) : Pointer;
      function    LoadFromFile ( filename : String ) : Pointer;
      function    Assign ( FromDyn : TrsDynarray ) : Pointer;
      property Count       : Longint   read FItemCount;
      property Size        : Longint   read FArraySize;
      property ItemSize    : Longint   read FItemSize;
      property DataPtr     : Pointer   read FPointer;
      property Items[Index: longint]: Pointer read GetItems; default;
    end;

implementation

{***********************}
{  Create & Initialize  }
{***********************}
constructor TrsDynArray.Create( ItemSize : Longint );
begin
  inherited create;
  if (ItemSize > 0) and (ItemSize < 65520) then begin
    FItemCount:= 0;
    FArraySize:= 0;
    FItemSize:= ItemSize;
    FPointer:= nil;
    end
  else raise EDynArrayCreateError.Create('Dynamic Array: Invalid Item Size');
end;

{***********************}
{    Destroy & Free     }
{***********************}
destructor TrsDynArray.Destroy;
begin
  Clear;
  FItemSize:= 0;
  inherited destroy;
end;

{***********************}
{      Clear Array      }
{***********************}
function TrsDynArray.Clear : pointer;
begin
  if FItemCount > 0 then begin
    FreeMem(FPointer, FArraySize);
    FItemCount:= 0;
    FArraySize:= 0;
    end;
  result:= NIL;
end;

{***********************}
{ Add an Array Element  }
{***********************}
function TrsDynArray.Add ( const Item ) : Pointer;
var P : Pointer;
begin
  if FItemSize > 0 then begin
    {Allocate next memory element}
    if FItemCount = 0 then GetMem( FPointer, FItemSize )
    else
    {$IFDEF Win32}
      ReAllocMem( FPointer, FArraySize + FItemSize );
    {$ELSE}
      FPointer:= ReAllocMem( FPointer, FArraySize, FArraySize + FItemSize );
    {$ENDIF}

    if FPointer <> nil then begin {check for valid pointer}
      {advance counters}
      inc(FItemCount);
      inc(FArraySize, FItemSize);
      {move data into array memory}
      P:= FPointer;
      inc( longint(P), (FItemSize * (FItemCount - 1)) );
      move( Item, P^, FItemSize );
      end
    else raise EDynArrayResources.Create('Dynamic Array: Out of resources during Add.');
    end
  else begin
    raise EDynArrayAddError.Create('Dynamic Array: Unable to add element.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{***********************}
{ Ins an Array Element  }
{***********************}
function TrsDynArray.Insert( Index: Longint; const Item ) : Pointer;
var
  x        : smallint;
  P1 : Pointer;
  P2 : Pointer;
begin
  if FItemCount > 0 then begin
    {resize array to open space for last item}

    {$IFDEF Win32}
      ReAllocMem( FPointer, FArraySize + FItemSize );
    {$ELSE}
      Fpointer:= ReAllocMem( FPointer, FArraySize, FArraySize + FItemSize );
    {$ENDIF}

    Inc(FArraySize, FItemSize);
    Inc(FItemCount);
    if (Index < FItemCount - 1) then begin
      {move items to open gap}
      P1:= FPointer;
      inc( longint(P1), FItemSize * (FItemCount - 2) );
      P2:= FPointer;
      inc( longint(P2), FItemSize * (FItemCount - 1) );
      for x:= FItemCount - 2 downto Index do begin
        move( P1^, P2^, FItemSize );
        dec( longint(P1), FItemSize );
        dec( longint(P2), FItemSize );
        end;
      {last, move index item to Index slot}
      P1:= FPointer;
      inc( longint(P1), FItemSize * (Index) );
      move( Item, P1^, FItemSize );
      end
      else if (Index = FItemCount - 2) then begin
      {They are inserting at the end}
      P1:= FPointer;
      inc( longint(P1), FItemSize * (Index) );
      move( Item, P1^, FItemSize );
      end
    else raise EDynArrayIndexBounds.Create('Dynamic Array: Index out of Bounds.');
  end
  else begin
    raise EDynArrayInsError.Create('Dynamic Array: Unable to insert element.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{**********************}
{ Del an Array Element }
{**********************}
function TrsDynArray.Delete( Index: Longint ) : Pointer;
var
  x        : smallint;
  P1 : Pointer;
  P2 : Pointer;
begin
  if FItemCount > 0 then begin
    if (Index < FItemCount - 1) then begin
      {move items to fill gap}
      P1:= FPointer;
      inc( longint(P1), FItemSize * Index );
      P2:= FPointer;
      inc( longint(P2), FItemSize * (Index + 1) );
      for x:= Index to FItemCount - 2 do begin
        move( P2^, P1^, FItemSize );
        inc( longint(P1), FItemSize );
        inc( longint(P2), FItemSize );
        end;
      end;
    {resize array to clip last item}

    {$IFDEF Win32}
      ReAllocMem( FPointer, FArraySize - FItemSize );
    {$ELSE}
      Fpointer:= ReAllocMem( FPointer, FArraySize, FArraySize - FItemSize );
    {$ENDIF}

    Dec(FArraySize, FItemSize);
    Dec(FItemCount);
    end
  else begin
    raise EDynArrayDelError.Create('Dynamic Array: Unable to delete element.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{*********************}
{ Save Array to File  }
{*********************}
function TrsDynArray.SaveToFile ( filename : String ) : Pointer;
var
  handle, x   : smallint;
  P     : Pointer;
begin
  if FItemCount > 0 then begin
    if fileexists( FileName ) then

      {$IFDEF Win32}
        DeleteFile( PChar(FileName) );
      {$ELSE}
        DeleteFile( FileName );
      {$ENDIF}

    handle:= FileCreate( FileName );
    if handle > - 1 then begin
      P := FPointer;
      for x:= 0 to FItemCount - 1 do begin
        FileWrite( handle, P^, FItemSize );
        inc( longint(P), FItemSize );
        end;
      FileClose(handle);
      end
    else raise EDynArrayFileError.Create('Dynamic Array: Unable to create file.');
    end
  else begin
    raise EDynArrayFileError.Create('Dynamic Array: No elements to save.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{*********************}
{Load Array from File }
{*********************}
function TrsDynArray.LoadFromFile ( filename : String ) : Pointer;
var
  handle      : smallint;
  tmpptr      : Pointer;
begin
  if FItemSize > 0 then begin
    if fileexists( FileName ) then begin
      if FItemCount > 0 then begin
        FreeMem(FPointer, FArraySize);
        FItemCount:= 0;
        FArraySize:= 0;
        FPointer:= nil;
        end;
      handle:= FileOpen( FileName, 0 );

      GetMem( tmpptr, FItemSize );

      while ( FileRead( handle, tmpptr^, FItemSize ) = FItemSize ) do
        FPointer:= Add(tmpptr^);
      FileClose(handle);
      {clean up}
      FreeMem( tmpptr, FItemSize );
      end
    else raise EDynArrayFileError.Create('Dynamic Array: File does not exist.');
  end
  else begin
    raise EDynArrayFileError.Create('Dynamic Array: Element size unknown.');
    FPointer:= nil;
    end;
  {return pointer}
  Result:= FPointer;
end;

{*********************}
{  Item Array Access  }
{*********************}
function TrsDynArray.GetItems(Index : Longint) : pointer;
var P : pointer;
begin
  if Index > FItemCount-1 then begin
    raise EDynArrayIndexBounds.Create('Dynamic Array: Index out of bounds.');
    Exit;
    end;
  P:= FPointer;
  inc(longint(P), longint(Index * FItemSize));
  Result:= P;
end;

function TrsDynArray.Assign ( FromDyn : TrsDynarray ) : Pointer;
begin
  if FromDyn.ItemSize = FItemSize then begin
    {Free the old array, if any}
    if FItemCount > 0 then begin
      FreeMem(FPointer, FArraySize);
      FItemCount:= 0;
      FArraySize:= 0;
      FPointer:= nil;
      end;
    {Create & Assign the array}
    FItemCount:= FromDyn.Count;
    FArraySize:= FromDyn.Size;
    GetMem( FPointer, FArraySize );
    move( FromDyn.DataPtr^, FPointer^, FArraySize );
    end
  else raise EDynArrayException.Create('Dynamic Array: Arrays not compatible for Assign.');
  Result:= FPointer;
end;

end.
